const fs = require('fs');

const SON_FILE = 'son.json';
const FINAL_FILE = 'final.json';
const OUTPUT_FILE = 'son_fixed.json';

try {
  console.log('Loading files...');
  const sonData = JSON.parse(fs.readFileSync(SON_FILE, 'utf8'));

  let finalData = [];
  try {
    finalData = JSON.parse(fs.readFileSync(FINAL_FILE, 'utf8'));
  } catch (e) {
    console.warn(
      `Warning: Could not load ${FINAL_FILE}. Merging will be skipped.`
    );
  }

  // Create lookup map from final.json
  const finalMap = new Map();
  if (Array.isArray(finalData)) {
    finalData.forEach((item) => {
      if (item.phrase) {
        finalMap.set(item.phrase.toLowerCase(), item);
      }
    });
  }

  console.log(`Loaded ${sonData.length} items from ${SON_FILE}`);
  console.log(`Loaded ${finalMap.size} unique items from ${FINAL_FILE}`);

  const fixedData = [];
  const seenPhrases = new Set();

  sonData.forEach((item, index) => {
    // 1. Regenerate ID
    item.id = index + 1;

    // 2. Merge missing data
    if (item.phrase) {
      const key = item.phrase.toLowerCase();
      const sourceItem = finalMap.get(key);

      if (sourceItem) {
        const fieldsToMerge = [
          'pronunciation',
          'audio',
          'definition',
          'mean',
          'synonyms',
          'sampleSentence',
          'sampleSentenceTr',
          'difficulty',
          'level',
        ];

        fieldsToMerge.forEach((field) => {
          if (
            !item[field] ||
            (Array.isArray(item[field]) && item[field].length === 0) ||
            item[field] === ''
          ) {
            if (sourceItem[field]) {
              item[field] = sourceItem[field];
            }
          }
        });
      }
    }

    // 3. Fill defaults for remaining missing fields
    if (!item.pronunciation) item.pronunciation = '';
    if (!item.audio) item.audio = '';
    if (!item.definition) item.definition = '';
    if (!item.mean) item.mean = '';
    if (!item.synonyms) item.synonyms = '';
    if (!item.sampleSentence) item.sampleSentence = [];
    if (!item.sampleSentenceTr) item.sampleSentenceTr = [];
    if (!item.sampleSentenceTr) item.sampleSentenceTr = ''; // Sometimes it's string in source, ensure consistency? Let's check target schema.
    // Wait, check_son.js expects sampleSentenceTr to be array usually.
    // Let's ensure array if the existing one is array-like?
    // The previous validation error "sampleSentenceTr is not an array" suggests it should be array.
    if (!Array.isArray(item.sampleSentence)) item.sampleSentence = [];
    if (!Array.isArray(item.sampleSentenceTr)) item.sampleSentenceTr = [];

    // PronunciationTr usually missing
    if (!item.pronunciationTr) item.pronunciationTr = '';

    // Defaults for level/difficulty if missing
    if (!item.level) item.level = 'A1';
    if (!item.difficulty) item.difficulty = 1;

    fixedData.push(item);
  });

  fs.writeFileSync(OUTPUT_FILE, JSON.stringify(fixedData, null, 2));
  console.log(`Successfully wrote ${fixedData.length} items to ${OUTPUT_FILE}`);
} catch (err) {
  console.error('Error:', err);
}
